<?php

namespace App\Http\Controllers;

use App\Models\Checklist;
use App\Models\Department;
use App\Models\Equipment;
use Illuminate\Http\Request;

class EquipmentController extends Controller
{
    public function index()
    {
        $equipment = Equipment::with(['department', 'checklists'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.equipment.index', compact('equipment'));
    }

    public function create()
    {
        $departments = Department::where('is_active', true)->get();
        $checklists = Checklist::where('is_active', true)->get();

        return view('admin.equipment.create', compact('departments', 'checklists'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'tag_id' => 'required|string|max:255|unique:equipment',
            'department_id' => 'required|exists:departments,id',
            'location' => 'required|string|max:255',
            'type' => 'required|in:tool,forklift,crane,vehicle,machinery,other',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'checklists' => 'array',
            'checklists.*' => 'exists:checklists,id',
        ]);

        $equipment = Equipment::create([
            'name' => $validated['name'],
            'tag_id' => $validated['tag_id'],
            'department_id' => $validated['department_id'],
            'location' => $validated['location'],
            'type' => $validated['type'],
            'description' => $validated['description'] ?? null,
            'is_active' => $request->boolean('is_active', true),
        ]);

        if (!empty($validated['checklists'])) {
            $equipment->checklists()->sync($validated['checklists']);
        }

        return redirect()->route('admin.equipment.index')
            ->with('success', 'Equipment created successfully.');
    }

    public function edit(Equipment $equipment)
    {
        $departments = Department::where('is_active', true)->get();
        $checklists = Checklist::where('is_active', true)->get();
        $selectedChecklists = $equipment->checklists->pluck('id')->toArray();

        return view('admin.equipment.edit', compact('equipment', 'departments', 'checklists', 'selectedChecklists'));
    }

    public function update(Request $request, Equipment $equipment)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'tag_id' => 'required|string|max:255|unique:equipment,tag_id,' . $equipment->id,
            'department_id' => 'required|exists:departments,id',
            'location' => 'required|string|max:255',
            'type' => 'required|in:tool,forklift,crane,vehicle,machinery,other',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'checklists' => 'array',
            'checklists.*' => 'exists:checklists,id',
        ]);

        $equipment->update([
            'name' => $validated['name'],
            'tag_id' => $validated['tag_id'],
            'department_id' => $validated['department_id'],
            'location' => $validated['location'],
            'type' => $validated['type'],
            'description' => $validated['description'] ?? null,
            'is_active' => $request->boolean('is_active', true),
        ]);

        $equipment->checklists()->sync($validated['checklists'] ?? []);

        return redirect()->route('admin.equipment.index')
            ->with('success', 'Equipment updated successfully.');
    }

    public function destroy(Equipment $equipment)
    {
        $equipment->delete();

        return redirect()->route('admin.equipment.index')
            ->with('success', 'Equipment deleted successfully.');
    }
}
